<?php
class SessionManager {
    private $sessionTimeout = 86400; // 24 heures
    
    public function __construct() {
        $this->initSession();
    }
    
    private function initSession() {
        if (session_status() === PHP_SESSION_NONE) {
            ini_set('session.cookie_httponly', 1);
            ini_set('session.cookie_secure', 1);
            ini_set('session.cookie_samesite', 'Strict');
            ini_set('session.gc_maxlifetime', $this->sessionTimeout);
            
            session_start();
            
            // Régénération périodique de l'ID de session
            if (!isset($_SESSION['created'])) {
                $_SESSION['created'] = time();
            } else if (time() - $_SESSION['created'] > 1800) {
                session_regenerate_id(true);
                $_SESSION['created'] = time();
            }
        }
    }
    
    public function login($username, $password, $rememberMe = false) {
        try {
            // Vérification des tentatives de connexion
            if ($this->isAccountLocked($username)) {
                throw new Exception('Compte temporairement verrouillé. Veuillez réessayer dans 15 minutes.');
            }
            
            // Recherche de l'administrateur
            $db = getDBConnection();
            $stmt = $db->prepare("
                SELECT id, username, email, password_hash, full_name, role, is_active, login_attempts
                FROM administrators 
                WHERE username = ? OR email = ?
            ");
            $stmt->execute([$username, $username]);
            $admin = $stmt->fetch();
            
            if (!$admin) {
                $this->logFailedAttempt($username);
                throw new Exception('Identifiants incorrects.');
            }
            
            if (!$admin['is_active']) {
                throw new Exception('Compte désactivé. Contactez le superadministrateur.');
            }
            
            // Vérification du mot de passe
            if (!password_verify($password, $admin['password_hash'])) {
                $this->logFailedAttempt($username, $admin['id']);
                throw new Exception('Identifiants incorrects.');
            }
            
            // Réinitialisation des tentatives échouées
            $this->resetFailedAttempts($admin['id']);
            
            // Création de la session
            $_SESSION['admin_id'] = $admin['id'];
            $_SESSION['admin_username'] = $admin['username'];
            $_SESSION['admin_email'] = $admin['email'];
            $_SESSION['admin_fullname'] = $admin['full_name'];
            $_SESSION['admin_role'] = $admin['role'];
            $_SESSION['admin_logged_in'] = true;
            $_SESSION['login_time'] = time();
            $_SESSION['session_token'] = bin2hex(random_bytes(32));
            
            // Mise à jour de la dernière connexion
            $this->updateLastLogin($admin['id']);
            
            // Journalisation
            $this->logAudit(
                $admin['id'],
                'login_success',
                'Connexion réussie',
                $_SERVER['REMOTE_ADDR']
            );
            
            return [
                'success' => true,
                'message' => 'Connexion réussie!',
                'user' => [
                    'id' => $admin['id'],
                    'username' => $admin['username'],
                    'full_name' => $admin['full_name'],
                    'role' => $admin['role']
                ]
            ];
            
        } catch (Exception $e) {
            $this->logAudit(
                null,
                'login_failed',
                'Échec connexion: ' . $e->getMessage(),
                $_SERVER['REMOTE_ADDR']
            );
            
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }
    
    public function logout() {
        $adminId = $_SESSION['admin_id'] ?? null;
        
        // Journalisation
        if ($adminId) {
            $this->logAudit(
                $adminId,
                'logout',
                'Déconnexion utilisateur',
                $_SERVER['REMOTE_ADDR']
            );
        }
        
        // Suppression de la session
        $_SESSION = [];
        
        // Suppression du cookie de session
        if (ini_get("session.use_cookies")) {
            $params = session_get_cookie_params();
            setcookie(session_name(), '', time() - 42000,
                $params["path"], $params["domain"],
                $params["secure"], $params["httponly"]
            );
        }
        
        session_destroy();
        
        return [
            'success' => true,
            'message' => 'Déconnexion réussie.'
        ];
    }
    
    public function isLoggedIn() {
        if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
            return false;
        }
        
        // Vérification du timeout de session
        if (time() - $_SESSION['login_time'] > $this->sessionTimeout) {
            $this->logout();
            return false;
        }
        
        // Vérification de l'activité de l'utilisateur en base
        if (!$this->isUserStillActive($_SESSION['admin_id'])) {
            $this->logout();
            return false;
        }
        
        // Mise à jour du temps d'activité
        $_SESSION['login_time'] = time();
        
        return true;
    }
    
    public function hasPermission($requiredRole = 'admin') {
        if (!$this->isLoggedIn()) {
            return false;
        }
        
        $userRole = $_SESSION['admin_role'];
        
        $rolesHierarchy = [
            'superadmin' => 2,
            'admin' => 1
        ];
        
        $userLevel = $rolesHierarchy[$userRole] ?? 0;
        $requiredLevel = $rolesHierarchy[$requiredRole] ?? 0;
        
        return $userLevel >= $requiredLevel;
    }
    
    private function isAccountLocked($username) {
        $db = getDBConnection();
        $stmt = $db->prepare("
            SELECT login_attempts, last_attempt 
            FROM administrators 
            WHERE username = ? OR email = ?
        ");
        $stmt->execute([$username, $username]);
        $admin = $stmt->fetch();
        
        if (!$admin) {
            return false;
        }
        
        // Vérifier si le compte est verrouillé (trop de tentatives)
        if ($admin['login_attempts'] >= 5) {
            $lockTime = strtotime($admin['last_attempt']);
            // Verrouillage pendant 15 minutes
            if (time() - $lockTime < 900) {
                return true;
            } else {
                // Réinitialiser après 15 minutes
                $this->resetFailedAttempts($admin['id']);
            }
        }
        
        return false;
    }
    
    private function logFailedAttempt($username, $adminId = null) {
        if ($adminId) {
            $db = getDBConnection();
            $stmt = $db->prepare("
                UPDATE administrators 
                SET login_attempts = login_attempts + 1, last_attempt = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$adminId]);
        }
        
        $this->logAudit(
            $adminId,
            'login_failed',
            'Tentative de connexion échouée pour: ' . $username,
            $_SERVER['REMOTE_ADDR']
        );
    }
    
    private function resetFailedAttempts($adminId) {
        $db = getDBConnection();
        $stmt = $db->prepare("
            UPDATE administrators 
            SET login_attempts = 0, last_attempt = NULL 
            WHERE id = ?
        ");
        $stmt->execute([$adminId]);
    }
    
    private function updateLastLogin($adminId) {
        $db = getDBConnection();
        $stmt = $db->prepare("
            UPDATE administrators 
            SET last_login = NOW() 
            WHERE id = ?
        ");
        $stmt->execute([$adminId]);
    }
    
    private function isUserStillActive($adminId) {
        $db = getDBConnection();
        $stmt = $db->prepare("
            SELECT is_active 
            FROM administrators 
            WHERE id = ?
        ");
        $stmt->execute([$adminId]);
        $admin = $stmt->fetch();
        
        return $admin && $admin['is_active'];
    }
    
    private function logAudit($adminId, $action, $description, $ipAddress) {
        $db = getDBConnection();
        $stmt = $db->prepare("
            INSERT INTO audit_logs 
            (admin_id, action, description, ip_address, user_agent) 
            VALUES (?, ?, ?, ?, ?)
        ");
        $stmt->execute([
            $adminId,
            $action,
            $description,
            $ipAddress,
            $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
    }
    
    public function getCurrentUser() {
        if (!$this->isLoggedIn()) {
            return null;
        }
        
        return [
            'id' => $_SESSION['admin_id'],
            'username' => $_SESSION['admin_username'],
            'email' => $_SESSION['admin_email'],
            'full_name' => $_SESSION['admin_fullname'],
            'role' => $_SESSION['admin_role']
        ];
    }
    
    public function getSessionToken() {
        return $_SESSION['session_token'] ?? null;
    }
}
?>