<?php
require_once 'includes/auth-check.php';
require_once '../config/database.php';

$sessionManager = new SessionManager();
$user = $sessionManager->getCurrentUser();
$db = getDBConnection();

// Actions sur les installations
$action = $_GET['action'] ?? '';
$installationId = $_GET['id'] ?? '';

// Suspension d'une installation
if ($action === 'suspend' && $installationId) {
    $stmt = $db->prepare("UPDATE installations SET status = 'suspended' WHERE id = ?");
    $stmt->execute([$installationId]);
    $success = "Installation suspendue avec succès";
}

// Réactivation d'une installation
if ($action === 'activate' && $installationId) {
    $stmt = $db->prepare("UPDATE installations SET status = 'active' WHERE id = ?");
    $stmt->execute([$installationId]);
    $success = "Installation réactivée avec succès";
}

// Récupération des installations avec filtres
$statusFilter = $_GET['status'] ?? '';
$searchTerm = $_GET['search'] ?? '';

$whereClause = "1=1";
$params = [];

if ($statusFilter) {
    $whereClause .= " AND i.status = ?";
    $params[] = $statusFilter;
}

if ($searchTerm) {
    $whereClause .= " AND (i.domain LIKE ? OR i.admin_email LIKE ? OR ik.client_name LIKE ?)";
    $searchTerm = "%$searchTerm%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$stmt = $db->prepare("
    SELECT 
        i.*,
        ik.key_name,
        ik.key_value,
        ik.client_name,
        ik.client_email,
        ik.institution_name,
        DATEDIFF(i.last_check, i.installed_at) as days_active
    FROM installations i
    INNER JOIN installation_keys ik ON i.key_id = ik.id
    WHERE $whereClause
    ORDER BY i.installed_at DESC
");

$stmt->execute($params);
$installations = $stmt->fetchAll();

// Statistiques des installations
$statsStmt = $db->query("
    SELECT 
        status,
        COUNT(*) as count,
        AVG(TIMESTAMPDIFF(DAY, installed_at, NOW())) as avg_days_active
    FROM installations 
    GROUP BY status
");
$installationStats = $statsStmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Installations - Pedazone Keys Manager</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.1.3/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <?php 
    $pageTitle = "Gestion des Installations";
    include 'includes/header.php'; 
    ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="fas fa-server me-2"></i>Installations
                    </h1>
                    <div class="btn-group">
                        <a href="reports.php?type=installations" class="btn btn-outline-primary">
                            <i class="fas fa-download me-2"></i>Exporter
                        </a>
                    </div>
                </div>

                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Statistiques rapides -->
                <div class="row mb-4">
                    <?php foreach ($installationStats as $stat): ?>
                    <div class="col-xl-3 col-md-6 mb-3">
                        <div class="card stat-card 
                            <?php echo match($stat['status']) {
                                'active' => 'bg-success text-white',
                                'suspended' => 'bg-warning text-dark',
                                'uninstalled' => 'bg-secondary text-white',
                                default => 'bg-light text-dark'
                            }; ?>">
                            <div class="card-body py-3">
                                <div class="d-flex justify-content-between">
                                    <div>
                                        <div class="small"><?php echo ucfirst($stat['status']); ?></div>
                                        <div class="h4 mb-0"><?php echo $stat['count']; ?></div>
                                    </div>
                                    <div class="align-self-center">
                                        <i class="fas fa-server fa-2x opacity-50"></i>
                                    </div>
                                </div>
                                <?php if ($stat['avg_days_active']): ?>
                                <small>Moyenne: <?php echo round($stat['avg_days_active']); ?> jours</small>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>

                <!-- Filtres et recherche -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label for="status" class="form-label">Statut</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="">Tous les statuts</option>
                                    <option value="active" <?php echo $statusFilter === 'active' ? 'selected' : ''; ?>>Actives</option>
                                    <option value="suspended" <?php echo $statusFilter === 'suspended' ? 'selected' : ''; ?>>Suspendues</option>
                                    <option value="uninstalled" <?php echo $statusFilter === 'uninstalled' ? 'selected' : ''; ?>>Désinstallées</option>
                                </select>
                            </div>
                            <div class="col-md-7">
                                <label for="search" class="form-label">Recherche</label>
                                <input type="text" class="form-control" id="search" name="search" 
                                       value="<?php echo htmlspecialchars($searchTerm); ?>" 
                                       placeholder="Rechercher par domaine, email ou client...">
                            </div>
                            <div class="col-md-2 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">
                                    <i class="fas fa-search me-2"></i>Filtrer
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Liste des installations -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-list me-2"></i>Liste des Installations (<?php echo count($installations); ?>)
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Domaine</th>
                                        <th>Client</th>
                                        <th>Environnement</th>
                                        <th>Statut</th>
                                        <th>Dates</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($installations as $installation): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($installation['domain']); ?></strong><br>
                                            <small class="text-muted"><?php echo htmlspecialchars($installation['admin_email']); ?></small>
                                        </td>
                                        <td>
                                            <?php echo htmlspecialchars($installation['client_name']); ?><br>
                                            <small class="text-muted">Clé: <?php echo substr($installation['key_value'], 0, 15); ?>...</small>
                                        </td>
                                        <td>
                                            <small>
                                                PHP: <?php echo htmlspecialchars($installation['php_version']); ?><br>
                                                Pedazone: <?php echo htmlspecialchars($installation['pedazone_version']); ?>
                                            </small>
                                        </td>
                                        <td>
                                            <?php
                                            $statusBadges = [
                                                'active' => ['bg-success', 'Actif'],
                                                'suspended' => ['bg-warning', 'Suspendu'],
                                                'uninstalled' => ['bg-secondary', 'Désinstallé']
                                            ];
                                            $statusInfo = $statusBadges[$installation['status']] ?? ['bg-secondary', 'Inconnu'];
                                            ?>
                                            <span class="badge <?php echo $statusInfo[0]; ?>">
                                                <?php echo $statusInfo[1]; ?>
                                            </span>
                                            <?php if ($installation['days_active']): ?>
                                            <br><small class="text-muted"><?php echo $installation['days_active']; ?> jours</small>
                                            <?php endif; ?>
                                        </td>
                                        <td>
                                            <small>
                                                Installé: <?php echo date('d/m/Y', strtotime($installation['installed_at'])); ?><br>
                                                Dernier check: 
                                                <?php 
                                                if ($installation['last_check']) {
                                                    echo date('d/m/Y', strtotime($installation['last_check']));
                                                } else {
                                                    echo '<span class="text-warning">Jamais</span>';
                                                }
                                                ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="installation-details.php?id=<?php echo $installation['id']; ?>" 
                                                   class="btn btn-outline-primary" title="Détails">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <?php if ($installation['status'] === 'active'): ?>
                                                <a href="installations.php?action=suspend&id=<?php echo $installation['id']; ?>" 
                                                   class="btn btn-outline-warning"
                                                   onclick="return confirm('Êtes-vous sûr de vouloir suspendre cette installation ?')"
                                                   title="Suspendre">
                                                    <i class="fas fa-pause"></i>
                                                </a>
                                                <?php elseif ($installation['status'] === 'suspended'): ?>
                                                <a href="installations.php?action=activate&id=<?php echo $installation['id']; ?>" 
                                                   class="btn btn-outline-success"
                                                   onclick="return confirm('Êtes-vous sûr de vouloir réactiver cette installation ?')"
                                                   title="Réactiver">
                                                    <i class="fas fa-play"></i>
                                                </a>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <?php if (empty($installations)): ?>
                        <div class="text-center py-4">
                            <i class="fas fa-server fa-3x text-muted mb-3"></i>
                            <p class="text-muted">Aucune installation trouvée</p>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.1.3/js/bootstrap.bundle.min.js"></script>
</body>
</html>