<?php
require_once 'includes/auth-check.php';
require_once '../config/database.php';

$sessionManager = new SessionManager();
$user = $sessionManager->getCurrentUser();
$db = getDBConnection();

// Actions sur les clés
$action = $_GET['action'] ?? '';
$keyId = $_GET['id'] ?? '';

// Génération d'une nouvelle clé
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['generate_key'])) {
    $keyName = $_POST['key_name'];
    $clientName = $_POST['client_name'];
    $clientEmail = $_POST['client_email'];
    $clientPhone = $_POST['client_phone'];
    $institutionType = $_POST['institution_type'];
    $institutionName = $_POST['institution_name'];
    $maxInstallations = $_POST['max_installations'];
    $expiryDays = $_POST['expiry_days'];
    $notes = $_POST['notes'];
    
    // Génération de la clé
    $keyValue = generateInstallationKey($clientName);
    $expiresAt = date('Y-m-d', strtotime("+$expiryDays days"));
    
    try {
        $stmt = $db->prepare("
            INSERT INTO installation_keys 
            (key_value, key_name, client_name, client_email, client_phone, institution_type, institution_name, max_installations, expires_at, notes, created_by)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            $keyValue,
            $keyName,
            $clientName,
            $clientEmail,
            $clientPhone,
            $institutionType,
            $institutionName,
            $maxInstallations,
            $expiresAt,
            $notes,
            $user['id']
        ]);
        
        $success = "Clé générée avec succès: <strong>$keyValue</strong>";
        
        // Journalisation
        $auditStmt = $db->prepare("
            INSERT INTO audit_logs (admin_id, action, description, ip_address, user_agent)
            VALUES (?, 'key_created', ?, ?, ?)
        ");
        $auditStmt->execute([
            $user['id'],
            "Création clé: $keyName ($keyValue) pour $clientName",
            $_SERVER['REMOTE_ADDR'],
            $_SERVER['HTTP_USER_AGENT'] ?? ''
        ]);
        
    } catch (Exception $e) {
        $error = "Erreur lors de la génération de la clé: " . $e->getMessage();
    }
}

// Révocation d'une clé
if ($action === 'revoke' && $keyId) {
    $stmt = $db->prepare("UPDATE installation_keys SET status = 'revoked' WHERE id = ?");
    $stmt->execute([$keyId]);
    $success = "Clé révoquée avec succès";
}

// Récupération des clés avec filtres
$statusFilter = $_GET['status'] ?? '';
$searchTerm = $_GET['search'] ?? '';

$whereClause = "1=1";
$params = [];

if ($statusFilter) {
    $whereClause .= " AND ik.status = ?";
    $params[] = $statusFilter;
}

if ($searchTerm) {
    $whereClause .= " AND (ik.key_name LIKE ? OR ik.client_name LIKE ? OR ik.key_value LIKE ?)";
    $searchTerm = "%$searchTerm%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$stmt = $db->prepare("
    SELECT 
        ik.*,
        a.username as created_by_name,
        COUNT(i.id) as current_installations
    FROM installation_keys ik
    LEFT JOIN administrators a ON ik.created_by = a.id
    LEFT JOIN installations i ON ik.id = i.key_id AND i.status = 'active'
    WHERE $whereClause
    GROUP BY ik.id
    ORDER BY ik.created_at DESC
");

$stmt->execute($params);
$keys = $stmt->fetchAll();
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gestion des Clés - Pedazone Keys Manager</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.1.3/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="fas fa-key me-2"></i>Gestion des Clés
                    </h1>
                    <a href="keys.php?action=create" class="btn btn-primary">
                        <i class="fas fa-plus me-2"></i>Nouvelle Clé
                    </a>
                </div>

                <?php if (isset($success)): ?>
                    <div class="alert alert-success alert-dismissible fade show" role="alert">
                        <?php echo $success; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <?php if (isset($error)): ?>
                    <div class="alert alert-danger alert-dismissible fade show" role="alert">
                        <?php echo $error; ?>
                        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                    </div>
                <?php endif; ?>

                <!-- Formulaire de création de clé -->
                <?php if ($action === 'create'): ?>
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-plus-circle me-2"></i>Générer une Nouvelle Clé
                        </h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="key_name" class="form-label">Nom de la Clé *</label>
                                        <input type="text" class="form-control" id="key_name" name="key_name" required>
                                        <div class="form-text">Nom identifiant cette clé (ex: "Université Paris - Licence Standard")</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="client_name" class="form-label">Nom du Client *</label>
                                        <input type="text" class="form-control" id="client_name" name="client_name" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="client_email" class="form-label">Email du Client *</label>
                                        <input type="email" class="form-control" id="client_email" name="client_email" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="client_phone" class="form-label">Téléphone</label>
                                        <input type="text" class="form-control" id="client_phone" name="client_phone">
                                    </div>
                                </div>
                                
                                <div class="col-md-6">
                                    <div class="mb-3">
                                        <label for="institution_type" class="form-label">Type d'Institution *</label>
                                        <select class="form-select" id="institution_type" name="institution_type" required>
                                            <option value="universite">Université</option>
                                            <option value="ecole">École</option>
                                            <option value="college">Collège</option>
                                            <option value="lycee">Lycée</option>
                                            <option value="autre">Autre</option>
                                        </select>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="institution_name" class="form-label">Nom de l'Institution</label>
                                        <input type="text" class="form-control" id="institution_name" name="institution_name">
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="max_installations" class="form-label">Nombre max d'Installations *</label>
                                        <input type="number" class="form-control" id="max_installations" name="max_installations" value="1" min="1" max="10" required>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label for="expiry_days" class="form-label">Validité (jours) *</label>
                                        <input type="number" class="form-control" id="expiry_days" name="expiry_days" value="30" min="1" max="365" required>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mb-3">
                                <label for="notes" class="form-label">Notes</label>
                                <textarea class="form-control" id="notes" name="notes" rows="3"></textarea>
                            </div>
                            
                            <div class="d-flex gap-2">
                                <button type="submit" name="generate_key" class="btn btn-success">
                                    <i class="fas fa-key me-2"></i>Générer la Clé
                                </button>
                                <a href="keys.php" class="btn btn-secondary">Annuler</a>
                            </div>
                        </form>
                    </div>
                </div>

                <?php else: ?>

                <!-- Filtres et recherche -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-4">
                                <label for="status" class="form-label">Statut</label>
                                <select class="form-select" id="status" name="status">
                                    <option value="">Tous les statuts</option>
                                    <option value="active" <?php echo $statusFilter === 'active' ? 'selected' : ''; ?>>Actives</option>
                                    <option value="used" <?php echo $statusFilter === 'used' ? 'selected' : ''; ?>>Utilisées</option>
                                    <option value="revoked" <?php echo $statusFilter === 'revoked' ? 'selected' : ''; ?>>Révoquées</option>
                                    <option value="expired" <?php echo $statusFilter === 'expired' ? 'selected' : ''; ?>>Expirées</option>
                                </select>
                            </div>
                            <div class="col-md-6">
                                <label for="search" class="form-label">Recherche</label>
                                <input type="text" class="form-control" id="search" name="search" 
                                       value="<?php echo htmlspecialchars($searchTerm); ?>" 
                                       placeholder="Rechercher par nom, client ou clé...">
                            </div>
                            <div class="col-md-2 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">
                                    <i class="fas fa-search me-2"></i>Filtrer
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Liste des clés -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-list me-2"></i>Liste des Clés (<?php echo count($keys); ?>)
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-hover">
                                <thead>
                                    <tr>
                                        <th>Clé</th>
                                        <th>Client</th>
                                        <th>Institution</th>
                                        <th>Statut</th>
                                        <th>Installations</th>
                                        <th>Expiration</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($keys as $key): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo htmlspecialchars($key['key_name']); ?></strong><br>
                                            <small class="text-muted"><?php echo htmlspecialchars($key['key_value']); ?></small>
                                        </td>
                                        <td>
                                            <?php echo htmlspecialchars($key['client_name']); ?><br>
                                            <small class="text-muted"><?php echo htmlspecialchars($key['client_email']); ?></small>
                                        </td>
                                        <td>
                                            <span class="badge bg-light text-dark"><?php echo ucfirst($key['institution_type']); ?></span><br>
                                            <small><?php echo htmlspecialchars($key['institution_name']); ?></small>
                                        </td>
                                        <td>
                                            <?php
                                            $statusBadges = [
                                                'active' => 'bg-success',
                                                'used' => 'bg-info', 
                                                'revoked' => 'bg-danger',
                                                'expired' => 'bg-warning'
                                            ];
                                            ?>
                                            <span class="badge <?php echo $statusBadges[$key['status']]; ?>">
                                                <?php echo ucfirst($key['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo $key['current_installations']; ?>/<?php echo $key['max_installations']; ?>
                                        </td>
                                        <td>
                                            <?php echo date('d/m/Y', strtotime($key['expires_at'])); ?><br>
                                            <small class="text-muted">
                                                <?php 
                                                $daysLeft = floor((strtotime($key['expires_at']) - time()) / (60 * 60 * 24));
                                                echo $daysLeft > 0 ? "$daysLeft jours" : "Expiré";
                                                ?>
                                            </small>
                                        </td>
                                        <td>
                                            <div class="btn-group btn-group-sm">
                                                <a href="key-details.php?id=<?php echo $key['id']; ?>" class="btn btn-outline-primary">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                                <?php if ($key['status'] === 'active'): ?>
                                                <a href="keys.php?action=revoke&id=<?php echo $key['id']; ?>" 
                                                   class="btn btn-outline-danger"
                                                   onclick="return confirm('Êtes-vous sûr de vouloir révoquer cette clé ?')">
                                                    <i class="fas fa-ban"></i>
                                                </a>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                        
                        <?php if (empty($keys)): ?>
                        <div class="text-center py-4">
                            <i class="fas fa-key fa-3x text-muted mb-3"></i>
                            <p class="text-muted">Aucune clé trouvée</p>
                            <a href="keys.php?action=create" class="btn btn-primary">Créer la première clé</a>
                        </div>
                        <?php endif; ?>
                    </div>
                </div>
                <?php endif; ?>
            </main>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.1.3/js/bootstrap.bundle.min.js"></script>
</body>
</html>