<?php
require_once 'includes/auth-check.php';
require_once '../config/database.php';

$sessionManager = new SessionManager();
$user = $sessionManager->getCurrentUser();
$db = getDBConnection();

// Type de rapport
$reportType = $_GET['type'] ?? 'overview';
$startDate = $_GET['start_date'] ?? date('Y-m-01'); // Début du mois
$endDate = $_GET['end_date'] ?? date('Y-m-d');
$export = isset($_GET['export']);

// Génération des rapports
$reportData = [];
$reportTitle = '';

switch ($reportType) {
    case 'overview':
        $reportTitle = 'Vue d\'Ensemble';
        
        // Statistiques générales
        $stmt = $db->query("SELECT * FROM view_keys_summary");
        $reportData['summary'] = $stmt->fetch();
        
        // Activité mensuelle
        $stmt = $db->prepare("
            SELECT 
                DATE_FORMAT(created_at, '%Y-%m') as month,
                COUNT(*) as keys_created,
                SUM(CASE WHEN status = 'used' THEN 1 ELSE 0 END) as keys_used
            FROM installation_keys 
            WHERE created_at BETWEEN ? AND ?
            GROUP BY DATE_FORMAT(created_at, '%Y-%m')
            ORDER BY month DESC
            LIMIT 12
        ");
        $stmt->execute([$startDate, $endDate]);
        $reportData['monthly_activity'] = $stmt->fetchAll();
        
        // Top clients
        $stmt = $db->query("
            SELECT 
                institution_type,
                institution_name,
                COUNT(*) as key_count,
                SUM(installations_count) as total_installations
            FROM installation_keys 
            GROUP BY institution_type, institution_name
            ORDER BY key_count DESC
            LIMIT 10
        ");
        $reportData['top_clients'] = $stmt->fetchAll();
        break;
        
    case 'installations':
        $reportTitle = 'Rapport des Installations';
        
        $stmt = $db->prepare("
            SELECT 
                i.*,
                ik.key_name,
                ik.client_name,
                ik.client_email,
                ik.institution_name,
                a.username as created_by
            FROM installations i
            INNER JOIN installation_keys ik ON i.key_id = ik.id
            LEFT JOIN administrators a ON ik.created_by = a.id
            WHERE i.installed_at BETWEEN ? AND ?
            ORDER BY i.installed_at DESC
        ");
        $stmt->execute([$startDate, $endDate]);
        $reportData['installations'] = $stmt->fetchAll();
        break;
        
    case 'keys':
        $reportTitle = 'Rapport des Clés';
        
        $stmt = $db->prepare("
            SELECT 
                ik.*,
                a.username as created_by,
                COUNT(i.id) as installation_count
            FROM installation_keys ik
            LEFT JOIN administrators a ON ik.created_by = a.id
            LEFT JOIN installations i ON ik.id = i.key_id
            WHERE ik.created_at BETWEEN ? AND ?
            GROUP BY ik.id
            ORDER BY ik.created_at DESC
        ");
        $stmt->execute([$startDate, $endDate]);
        $reportData['keys'] = $stmt->fetchAll();
        break;
        
    case 'audit':
        $reportTitle = 'Journal d\'Audit';
        
        $stmt = $db->prepare("
            SELECT 
                al.*,
                a.username as admin_name
            FROM audit_logs al
            LEFT JOIN administrators a ON al.admin_id = a.id
            WHERE al.created_at BETWEEN ? AND ?
            ORDER BY al.created_at DESC
            LIMIT 500
        ");
        $stmt->execute([$startDate, $endDate]);
        $reportData['audit_logs'] = $stmt->fetchAll();
        break;
}

// Export des données
if ($export) {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="pedazone_report_' . $reportType . '_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    switch ($reportType) {
        case 'installations':
            fputcsv($output, ['Domaine', 'Client', 'Email Admin', 'Statut', 'Date Installation', 'Version PHP', 'Version Pedazone']);
            foreach ($reportData['installations'] as $row) {
                fputcsv($output, [
                    $row['domain'],
                    $row['client_name'],
                    $row['admin_email'],
                    $row['status'],
                    $row['installed_at'],
                    $row['php_version'],
                    $row['pedazone_version']
                ]);
            }
            break;
            
        case 'keys':
            fputcsv($output, ['Clé', 'Nom', 'Client', 'Email', 'Statut', 'Installations', 'Expiration', 'Créée le']);
            foreach ($reportData['keys'] as $row) {
                fputcsv($output, [
                    $row['key_value'],
                    $row['key_name'],
                    $row['client_name'],
                    $row['client_email'],
                    $row['status'],
                    $row['installation_count'],
                    $row['expires_at'],
                    $row['created_at']
                ]);
            }
            break;
    }
    
    fclose($output);
    exit;
}
?>
<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Rapports - Pedazone Keys Manager</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.1.3/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <?php 
    $pageTitle = "Rapports et Statistiques";
    include 'includes/header.php'; 
    ?>
    
    <div class="container-fluid">
        <div class="row">
            <?php include 'includes/sidebar.php'; ?>
            
            <main class="col-md-9 ms-sm-auto col-lg-10 px-md-4">
                <div class="d-flex justify-content-between flex-wrap flex-md-nowrap align-items-center pt-3 pb-2 mb-3 border-bottom">
                    <h1 class="h2">
                        <i class="fas fa-chart-bar me-2"></i><?php echo $reportTitle; ?>
                    </h1>
                    <div class="btn-group">
                        <?php if (in_array($reportType, ['installations', 'keys', 'audit'])): ?>
                        <a href="reports.php?type=<?php echo $reportType; ?>&start_date=<?php echo $startDate; ?>&end_date=<?php echo $endDate; ?>&export=1" 
                           class="btn btn-success">
                            <i class="fas fa-file-csv me-2"></i>Exporter CSV
                        </a>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Filtres de période -->
                <div class="card mb-4">
                    <div class="card-body">
                        <form method="GET" class="row g-3">
                            <div class="col-md-3">
                                <label for="type" class="form-label">Type de Rapport</label>
                                <select class="form-select" id="type" name="type" onchange="this.form.submit()">
                                    <option value="overview" <?php echo $reportType === 'overview' ? 'selected' : ''; ?>>Vue d'ensemble</option>
                                    <option value="installations" <?php echo $reportType === 'installations' ? 'selected' : ''; ?>>Installations</option>
                                    <option value="keys" <?php echo $reportType === 'keys' ? 'selected' : ''; ?>>Clés d'installation</option>
                                    <option value="audit" <?php echo $reportType === 'audit' ? 'selected' : ''; ?>>Journal d'audit</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label for="start_date" class="form-label">Date de début</label>
                                <input type="date" class="form-control" id="start_date" name="start_date" 
                                       value="<?php echo $startDate; ?>">
                            </div>
                            <div class="col-md-3">
                                <label for="end_date" class="form-label">Date de fin</label>
                                <input type="date" class="form-control" id="end_date" name="end_date" 
                                       value="<?php echo $endDate; ?>">
                            </div>
                            <div class="col-md-3 d-flex align-items-end">
                                <button type="submit" class="btn btn-primary w-100">
                                    <i class="fas fa-filter me-2"></i>Appliquer
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Contenu du rapport -->
                <div class="card">
                    <div class="card-body">
                        <?php if ($reportType === 'overview'): ?>
                        
                        <!-- Vue d'ensemble avec graphiques -->
                        <div class="row mb-5">
                            <div class="col-md-6">
                                <h5>Statistiques Générales</h5>
                                <div class="table-responsive">
                                    <table class="table table-bordered">
                                        <tr>
                                            <td><strong>Clés Totales</strong></td>
                                            <td><?php echo $reportData['summary']['total_keys']; ?></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Clés Actives</strong></td>
                                            <td><span class="badge bg-success"><?php echo $reportData['summary']['active_keys']; ?></span></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Clés Utilisées</strong></td>
                                            <td><span class="badge bg-info"><?php echo $reportData['summary']['used_keys']; ?></span></td>
                                        </tr>
                                        <tr>
                                            <td><strong>Installations Total</strong></td>
                                            <td><?php echo $reportData['summary']['total_installations']; ?></td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <h5>Activité Mensuelle</h5>
                                <canvas id="monthlyChart" height="200"></canvas>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <h5>Top Clients</h5>
                                <div class="table-responsive">
                                    <table class="table table-striped">
                                        <thead>
                                            <tr>
                                                <th>Institution</th>
                                                <th>Type</th>
                                                <th>Clés Générées</th>
                                                <th>Installations</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($reportData['top_clients'] as $client): ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($client['institution_name'] ?: 'Non spécifié'); ?></td>
                                                <td><span class="badge bg-light text-dark"><?php echo ucfirst($client['institution_type']); ?></span></td>
                                                <td><?php echo $client['key_count']; ?></td>
                                                <td><?php echo $client['total_installations']; ?></td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <?php elseif ($reportType === 'installations'): ?>
                        
                        <!-- Rapport des installations -->
                        <h5>Installations (<?php echo count($reportData['installations']); ?>)</h5>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Domaine</th>
                                        <th>Client</th>
                                        <th>Environnement</th>
                                        <th>Statut</th>
                                        <th>Date Installation</th>
                                        <th>Créée par</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($reportData['installations'] as $installation): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($installation['domain']); ?></td>
                                        <td><?php echo htmlspecialchars($installation['client_name']); ?></td>
                                        <td>
                                            <small>PHP: <?php echo $installation['php_version']; ?><br>
                                            Pedazone: <?php echo $installation['pedazone_version']; ?></small>
                                        </td>
                                        <td>
                                            <span class="badge <?php echo match($installation['status']) {
                                                'active' => 'bg-success',
                                                'suspended' => 'bg-warning',
                                                default => 'bg-secondary'
                                            }; ?>">
                                                <?php echo ucfirst($installation['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo date('d/m/Y H:i', strtotime($installation['installed_at'])); ?></td>
                                        <td><?php echo htmlspecialchars($installation['created_by']); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <?php elseif ($reportType === 'keys'): ?>
                        
                        <!-- Rapport des clés -->
                        <h5>Clés d'Installation (<?php echo count($reportData['keys']); ?>)</h5>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Clé</th>
                                        <th>Nom</th>
                                        <th>Client</th>
                                        <th>Statut</th>
                                        <th>Installations</th>
                                        <th>Expiration</th>
                                        <th>Créée par</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($reportData['keys'] as $key): ?>
                                    <tr>
                                        <td><code><?php echo htmlspecialchars($key['key_value']); ?></code></td>
                                        <td><?php echo htmlspecialchars($key['key_name']); ?></td>
                                        <td><?php echo htmlspecialchars($key['client_name']); ?></td>
                                        <td>
                                            <span class="badge <?php echo match($key['status']) {
                                                'active' => 'bg-success',
                                                'used' => 'bg-info',
                                                'revoked' => 'bg-danger',
                                                'expired' => 'bg-warning',
                                                default => 'bg-secondary'
                                            }; ?>">
                                                <?php echo ucfirst($key['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo $key['installation_count']; ?>/<?php echo $key['max_installations']; ?></td>
                                        <td><?php echo date('d/m/Y', strtotime($key['expires_at'])); ?></td>
                                        <td><?php echo htmlspecialchars($key['created_by']); ?></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <?php elseif ($reportType === 'audit'): ?>
                        
                        <!-- Journal d'audit -->
                        <h5>Journal d'Audit (<?php echo count($reportData['audit_logs']); ?> entrées)</h5>
                        <div class="table-responsive">
                            <table class="table table-striped">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Utilisateur</th>
                                        <th>Action</th>
                                        <th>Description</th>
                                        <th>IP</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($reportData['audit_logs'] as $log): ?>
                                    <tr>
                                        <td><?php echo date('d/m/Y H:i', strtotime($log['created_at'])); ?></td>
                                        <td><?php echo htmlspecialchars($log['admin_name'] ?: 'Système'); ?></td>
                                        <td><span class="badge bg-secondary"><?php echo $log['action']; ?></span></td>
                                        <td><?php echo htmlspecialchars($log['description']); ?></td>
                                        <td><small><?php echo $log['ip_address']; ?></small></td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <?php endif; ?>
                    </div>
                </div>
            </main>
        </div>
    </div>

    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.1.3/js/bootstrap.bundle.min.js"></script>
    <script>
        <?php if ($reportType === 'overview' && !empty($reportData['monthly_activity'])): ?>
        // Graphique d'activité mensuelle
        const monthlyCtx = document.getElementById('monthlyChart').getContext('2d');
        const monthlyChart = new Chart(monthlyCtx, {
            type: 'bar',
            data: {
                labels: [<?php echo implode(',', array_map(function($item) { 
                    return "'" . date('M Y', strtotime($item['month'] . '-01')) . "'"; 
                }, $reportData['monthly_activity'])); ?>],
                datasets: [{
                    label: 'Clés Créées',
                    data: [<?php echo implode(',', array_column($reportData['monthly_activity'], 'keys_created')); ?>],
                    backgroundColor: 'rgba(102, 126, 234, 0.8)'
                }, {
                    label: 'Clés Utilisées',
                    data: [<?php echo implode(',', array_column($reportData['monthly_activity'], 'keys_used')); ?>],
                    backgroundColor: 'rgba(16, 185, 129, 0.8)'
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
        <?php endif; ?>
    </script>
</body>
</html>