<?php
require_once '../config/config.php';

class KeyValidationAPI {
    private $db;
    private $requestData;
    
    public function __construct() {
        $this->db = getDBConnection();
        $this->getRequestData();
        $this->logRequest();
    }
    
    private function getRequestData() {
        $this->requestData = [
            'installation_key' => $_POST['installation_key'] ?? $_GET['installation_key'] ?? '',
            'domain' => $_POST['domain'] ?? $_GET['domain'] ?? '',
            'client_email' => $_POST['client_email'] ?? $_GET['client_email'] ?? '',
            'action' => $_POST['action'] ?? $_GET['action'] ?? 'validate',
            'ip_address' => $this->getClientIP()
        ];
    }
    
    public function handleRequest() {
        try {
            if (empty($this->requestData['installation_key'])) {
                throw new Exception('Clé d\'installation manquante.');
            }
            
            $this->checkRateLimit();
            
            switch ($this->requestData['action']) {
                case 'validate':
                    $result = $this->validateKey();
                    break;
                case 'activate':
                    $result = $this->activateKey();
                    break;
                default:
                    throw new Exception('Action non supportée.');
            }
            
            $this->logResponse(200, $result);
            echo $this->formatResponse(true, 'Validation réussie', $result);
            
        } catch (Exception $e) {
            $this->logResponse(400, ['error' => $e->getMessage()]);
            echo $this->formatResponse(false, $e->getMessage(), null, 400);
        }
    }
    
    private function validateKey() {
        $key = $this->requestData['installation_key'];
        
        $stmt = $this->db->prepare("
            SELECT 
                ik.*,
                a.username as created_by_name,
                COUNT(i.id) as current_installations
            FROM installation_keys ik
            LEFT JOIN administrators a ON ik.created_by = a.id
            LEFT JOIN installations i ON ik.id = i.key_id AND i.status = 'active'
            WHERE ik.key_value = ? AND ik.status = 'active'
            GROUP BY ik.id
        ");
        $stmt->execute([$key]);
        $keyData = $stmt->fetch();
        
        if (!$keyData) {
            throw new Exception('Clé d\'installation invalide ou introuvable.');
        }
        
        if (strtotime($keyData['expires_at']) < time()) {
            $this->markKeyAsExpired($keyData['id']);
            throw new Exception('Clé d\'installation expirée.');
        }
        
        if ($keyData['current_installations'] >= $keyData['max_installations']) {
            throw new Exception('Nombre maximum d\'installations atteint pour cette clé.');
        }
        
        return [
            'valid' => true,
            'key_name' => $keyData['key_name'],
            'client_name' => $keyData['client_name'],
            'client_email' => $keyData['client_email'],
            'institution_name' => $keyData['institution_name'],
            'max_installations' => (int)$keyData['max_installations'],
            'current_installations' => (int)$keyData['current_installations'],
            'expires_at' => $keyData['expires_at'],
            'created_at' => $keyData['created_at']
        ];
    }
    
    private function activateKey() {
        $validation = $this->validateKey();
        
        if (!$validation['valid']) {
            throw new Exception('Clé invalide, activation impossible.');
        }
        
        $stmt = $this->db->prepare("
            INSERT INTO installations 
            (key_id, domain, server_ip, admin_email, pedazone_version, php_version, database_type, web_server, installation_data)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $keyId = $this->getKeyId($this->requestData['installation_key']);
        $installationData = json_encode([
            'domain' => $this->requestData['domain'],
            'client_email' => $this->requestData['client_email'],
            'server_info' => $_SERVER,
            'installed_at' => date('c')
        ]);
        
        $stmt->execute([
            $keyId,
            $this->requestData['domain'],
            $this->requestData['ip_address'],
            $this->requestData['client_email'],
            '1.0.0',
            PHP_VERSION,
            'MySQL',
            $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
            $installationData
        ]);
        
        $this->incrementInstallationCount($keyId);
        
        return [
            'activated' => true,
            'installation_id' => $this->db->lastInsertId(),
            'domain' => $this->requestData['domain'],
            'activated_at' => date('c')
        ];
    }
    
    private function checkRateLimit() {
        $ip = $this->requestData['ip_address'];
        $timeframe = time() - 3600;
        
        $stmt = $this->db->prepare("
            SELECT COUNT(*) as request_count 
            FROM api_requests 
            WHERE ip_address = ? AND created_at > FROM_UNIXTIME(?)
        ");
        $stmt->execute([$ip, $timeframe]);
        $result = $stmt->fetch();
        
        if ($result['request_count'] >= 100) {
            throw new Exception('Limite de requêtes API atteinte. Veuillez réessayer plus tard.');
        }
    }
    
    private function markKeyAsExpired($keyId) {
        $stmt = $this->db->prepare("
            UPDATE installation_keys 
            SET status = 'expired' 
            WHERE id = ? AND status = 'active'
        ");
        $stmt->execute([$keyId]);
    }
    
    private function incrementInstallationCount($keyId) {
        $stmt = $this->db->prepare("
            UPDATE installation_keys 
            SET installations_count = installations_count + 1,
                used_at = NOW()
            WHERE id = ?
        ");
        $stmt->execute([$keyId]);
    }
    
    private function getKeyId($keyValue) {
        $stmt = $this->db->prepare("SELECT id FROM installation_keys WHERE key_value = ?");
        $stmt->execute([$keyValue]);
        $result = $stmt->fetch();
        return $result ? $result['id'] : null;
    }
    
    private function logRequest() {
        $stmt = $this->db->prepare("
            INSERT INTO api_requests 
            (api_key, endpoint, installation_key, ip_address, user_agent, request_method, response_code, response_time)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ");
        
        $stmt->execute([
            'API-KEY',
            'validate-key',
            $this->requestData['installation_key'],
            $this->requestData['ip_address'],
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $_SERVER['REQUEST_METHOD'] ?? 'GET',
            0,
            0
        ]);
        
        $this->requestLogId = $this->db->lastInsertId();
        $this->startTime = microtime(true);
    }
    
    private function logResponse($responseCode, $responseData) {
        $responseTime = microtime(true) - $this->startTime;
        
        $stmt = $this->db->prepare("
            UPDATE api_requests 
            SET response_code = ?, response_time = ?
            WHERE id = ?
        ");
        $stmt->execute([$responseCode, $responseTime, $this->requestLogId]);
    }
    
    private function formatResponse($success, $message, $data = null, $code = 200) {
        http_response_code($code);
        return json_encode([
            'success' => $success,
            'message' => $message,
            'data' => $data,
            'timestamp' => date('c'),
            'api_version' => '1.0'
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }
    
    private function getClientIP() {
        if (!empty($_SERVER['HTTP_CLIENT_IP'])) {
            return $_SERVER['HTTP_CLIENT_IP'];
        } elseif (!empty($_SERVER['HTTP_X_FORWARDED_FOR'])) {
            return $_SERVER['HTTP_X_FORWARDED_FOR'];
        } else {
            return $_SERVER['REMOTE_ADDR'];
        }
    }
}

// Headers CORS
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Exécution
try {
    $api = new KeyValidationAPI();
    $api->handleRequest();
} catch (Exception $e) {
    error_log("Erreur API Key Validation: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Erreur interne du serveur',
        'timestamp' => date('c')
    ]);
}
?>